#!/bin/bash

echo "================================================"
echo "Raspberry Pi AI HAT Full Installer + Object Detection"
echo "================================================"

# -----------------------------
# 1️⃣ System update
# -----------------------------
echo "🔄 Updating system..."
sudo apt update
sudo apt full-upgrade -y
sudo rpi-eeprom-update -a

# -----------------------------
# 2️⃣ Install AI HAT and camera packages
# -----------------------------
echo ""
echo "📦 Installing AI HAT and camera packages..."

# Install dkms
sudo apt install -y dkms  # Ensures kernel modules can be built dynamically

# Check if hailo-h10-all is installed
if dpkg -s hailo-h10-all &> /dev/null; then
    echo "✅ hailo-h10-all is already installed. Skipping..."
else
    echo "⬇️ hailo-h10-all not found. Installing..."
    sudo apt install -y hailo-h10-all
fi

# Install rpicam packages
sudo apt install -y rpicam-apps  # Raspberry Pi camera apps
sudo apt install -y rpicam-apps-hailo-postprocess  # Hailo post-processing files

# -----------------------------
# 3️⃣ Verification checks
# -----------------------------
echo ""
echo "🔍 Running verification checks..."

echo ""
echo "➡️ OS VERSION"
cat /etc/os-release

echo ""
echo "➡️ HAILO DEVICE CHECK"
hailortcli fw-control identify || echo "❌ Hailo device not detected"

echo ""
echo "➡️ CAMERA CHECK (2 seconds preview)"
rpicam-hello -t 2000 || echo "❌ Camera preview failed"

echo ""
echo "➡️ HAILO POST-PROCESS FILES"
ls /usr/share/rpi-camera-assets | grep hailo || echo "❌ No Hailo files found"

# -----------------------------
# 4️⃣ Object detection prompt
# -----------------------------
echo ""
read -rp "Do you want to start object detection now? (Y/N): " start_detection

if [[ "$start_detection" =~ ^([Yy]|[Yy][Ee][Ss])$ ]]; then
    ROTATION=0
    read -rp "Do you want to rotate the camera? (Y/N): " rotate_answer
    if [[ "$rotate_answer" =~ ^([Yy]|[Yy][Ee][Ss])$ ]]; then
        while true; do
            read -rp "Enter rotation in degrees (90, 180, 270): " ROT
            if [[ "$ROT" == "90" || "$ROT" == "180" || "$ROT" == "270" ]]; then
                ROTATION=$ROT
                break
            else
                echo "❌ Invalid input. Please enter 90, 180, or 270."
            fi
        done
    fi

    echo ""
    echo "➡️ Starting object detection with rotation: $ROTATION°"
    echo "Press Ctrl+C to exit the live feed."

    DISPLAY=:0 rpicam-hello -t 0 \
      --post-process-file /usr/share/rpi-camera-assets/hailo_yolov8_inference.json \
      --lores-width 640 --lores-height 640 \
      --rotation "$ROTATION"
else
    echo "❌ Object detection skipped. Installation and verification complete!"
fi

echo ""
echo "✅ Script finished!"
